import tkinter as tk
import random
import sys

# Constants
SCREEN_WIDTH = 1080
SCREEN_HEIGHT = 400
GROUND_HEIGHT = 100
PLAYER_SIZE = 35
JUMP_VELOCITY = -19
GRAVITY = 2.0
SCROLL_SPEED = 8.7
FPS = 150
FRAME_TIME = 1000 // FPS

# Platform constants
PLATFORM_WIDTH = 100
PLATFORM_HEIGHT = 20
PLATFORM_GAP = 250
PLATFORM_MIN_Y = 150
PLATFORM_MAX_Y = SCREEN_HEIGHT - GROUND_HEIGHT - 100

# Colors
WHITE = "#FF0000"
BLACK = "#000000"
BLUE = "#0008FF"
GREEN = "#14B03D"
DARK_GREEN = "#00FF1A"
RED = "#C83232"
DARK_RED = "#641919"
GRAY = "#AAAAAA"

class Game:
    def __init__(self, root):
        self.root = root
        self.root.title("Geometry Dash Clone")
        self.canvas = tk.Canvas(root, width=SCREEN_WIDTH, height=SCREEN_HEIGHT, bg=BLUE)
        self.canvas.pack()

        self.root.bind("<space>", self.jump)

        self.player_x = 100
        self.player_y = SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE
        self.player_velocity_y = 0
        self.on_ground = True
        self.jump_count = 0

        self.spikes = [self.create_spike(SCREEN_WIDTH + i * 200) for i in range(3)]
        self.platforms = [self.create_platform(SCREEN_WIDTH + i * PLATFORM_GAP) for i in range(5)]

        self.score = 0
        self.game_over = False

        self.player = self.canvas.create_rectangle(
            self.player_x, self.player_y,
            self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE,
            fill=WHITE, outline=BLACK, width=2
        )
        self.score_text = self.canvas.create_text(
            10, 10, text="Score: 0", anchor="nw", fill=WHITE, font=("Arial", 25)
        )

        self.update()

    def create_spike(self, x, y=None):
        if y is None:
            y = SCREEN_HEIGHT - GROUND_HEIGHT
        shadow = self.canvas.create_polygon(
            x + 2, y + 2,
            x + 22, y + 2,
            x + 12, y - 28,
            fill=DARK_RED, outline=BLACK
        )
        spike = self.canvas.create_polygon(
            x, y,
            x + 20, y,
            x + 10, y - 30,
            fill=RED, outline=BLACK
        )
        return {"x": x, "y": y, "shadow": shadow, "spike": spike}

    def create_platform(self, x):
        y = random.randint(PLATFORM_MIN_Y, PLATFORM_MAX_Y)
        platform = self.canvas.create_rectangle(
            x, y, x + PLATFORM_WIDTH, y + PLATFORM_HEIGHT,
            fill=GRAY, outline=BLACK
        )
        return {"x": x, "y": y, "rect": platform}

    def jump(self, event):
        if self.jump_count < 2 and not self.game_over:
            self.player_velocity_y = JUMP_VELOCITY
            self.jump_count += 1
            self.on_ground = False

    def update(self):
        try:
            if not self.game_over:
                self.player_velocity_y += GRAVITY
                self.player_y += self.player_velocity_y

                # Deadly ground collision
                if self.player_y >= SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE:
                    self.end_game()
                    return

                self.canvas.coords(
                    self.player,
                    self.player_x, self.player_y,
                    self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE
                )

                for spike in self.spikes[:]:
                    spike["x"] -= SCROLL_SPEED
                    self.canvas.move(spike["spike"], -SCROLL_SPEED, 0)
                    self.canvas.move(spike["shadow"], -SCROLL_SPEED, 0)
                    if spike["x"] < -20:
                        self.canvas.delete(spike["spike"])
                        self.canvas.delete(spike["shadow"])
                        self.spikes.remove(spike)
                        self.spikes.append(self.create_spike(SCREEN_WIDTH + random.randint(400, 550)))

                    if self.check_collision(spike["x"], spike["y"], 20, 30):
                        self.end_game()
                        return

                for platform in self.platforms[:]:
                    platform["x"] -= SCROLL_SPEED
                    self.canvas.move(platform["rect"], -SCROLL_SPEED, 0)

                    if platform["x"] < -PLATFORM_WIDTH:
                        self.canvas.delete(platform["rect"])
                        self.platforms.remove(platform)
                        self.platforms.append(self.create_platform(SCREEN_WIDTH + random.randint(20, 200)))

                    px1, py1 = self.player_x, self.player_y
                    px2, py2 = self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE
                    plat_x1, plat_y1 = platform["x"], platform["y"]
                    plat_x2, plat_y2 = platform["x"] + PLATFORM_WIDTH, platform["y"] + PLATFORM_HEIGHT

                    if (px2 > plat_x1 and px1 < plat_x2 and
                        py2 >= plat_y1 and py2 <= plat_y1 + 20 and
                        self.player_velocity_y >= 0):
                        self.player_y = plat_y1 - PLAYER_SIZE
                        self.player_velocity_y = 0
                        self.on_ground = True
                        self.jump_count = 0

                self.score += 1
                self.canvas.itemconfig(self.score_text, text=f"Score: {self.score}")

            self.canvas.delete("ground")
            for x in range(0, SCREEN_WIDTH, 20):
                for y in range(SCREEN_HEIGHT - GROUND_HEIGHT, SCREEN_HEIGHT, 20):
                    color = GREEN if (x // 20 + y // 20) % 2 == 0 else DARK_GREEN
                    self.canvas.create_rectangle(
                        x, y, x + 20, y + 20, fill=color, outline="", tags="ground"
                    )

            if self.game_over:
                self.root.bind("<r>", self.restart)

            self.root.after(FRAME_TIME, self.update)

        except Exception as e:
            print(f"Game loop error: {e}")
            self.root.quit()

    def check_collision(self, obj_x, obj_y, obj_w, obj_h):
        px1, py1 = self.player_x, self.player_y
        px2, py2 = self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE
        ox1, oy1 = obj_x, obj_y - obj_h
        ox2, oy2 = obj_x + obj_w, obj_y
        return px2 > ox1 and px1 < ox2 and py2 > oy1 and py1 < oy2

    def end_game(self):
        self.game_over = True
        self.canvas.create_text(
            SCREEN_WIDTH // 2, SCREEN_HEIGHT // 2,
            text="Game Over! Press R to Restart", fill=WHITE, font=("Arial", 30)
        )

    def restart(self, event):
        try:
            self.canvas.delete("all")
            self.player_x = 100
            self.player_y = SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE
            self.player_velocity_y = 0
            self.on_ground = True
            self.jump_count = 0

            for spike in self.spikes:
                self.canvas.delete(spike["spike"])
                self.canvas.delete(spike["shadow"])
            for platform in self.platforms:
                self.canvas.delete(platform["rect"])

            self.spikes = [self.create_spike(SCREEN_WIDTH + i * 200) for i in range(3)]
            self.platforms = [self.create_platform(SCREEN_WIDTH + i * PLATFORM_GAP) for i in range(5)]
            self.score = 0
            self.game_over = False

            self.player = self.canvas.create_rectangle(
                self.player_x, self.player_y,
                self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE,
                fill=WHITE, outline=BLACK, width=2
            )
            self.score_text = self.canvas.create_text(
                10, 10, text="Score: 0", anchor="nw", fill=WHITE, font=("Arial", 25)
            )
            self.root.bind("<r>", lambda e: None)
        except Exception as e:
            print(f"Restart error: {e}")

def main():
    try:
        root = tk.Tk()
        game = Game(root)
        root.mainloop()
    except Exception as e:
        print(f"Main error: {e}")
        sys.exit()

if __name__ == "__main__":
    main()
