import tkinter as tk
import random
import sys

# Constants
SCREEN_WIDTH = 700
SCREEN_HEIGHT = 400
GROUND_HEIGHT = 100
PLAYER_SIZE = 30
JUMP_VELOCITY = -25              
GRAVITY = 2.5
SCROLL_SPEED = 18
FPS = 280
FRAME_TIME = 1000 // FPS  # Milliseconds per frame

# Colors for textures
WHITE = "#FF0000"
BLACK = "#000000"
BLUE = "#000000"  # Background
GREEN = "#AA1919"  # Ground
DARK_GREEN = "#EB1212"  # Ground texture
RED = "#C83232"  # Spikes
DARK_RED = "#641919"  # Spike shadow

class Game:
    def __init__(self, root):
        self.root = root
        self.root.title("Geometry Dash Clone")
        self.canvas = tk.Canvas(root, width=SCREEN_WIDTH, height=SCREEN_HEIGHT, bg=BLUE)
        self.canvas.pack()
        
        # Bind Space key for jumping
        self.root.bind("<space>", self.jump)
        
        # Game state
        self.player_x = 100
        self.player_y = SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE
        self.player_velocity_y = 0
        self.on_ground = True
        self.spikes = [self.create_spike(SCREEN_WIDTH + i * 200) for i in range(9)]
        self.score = 0
        self.game_over = False
        
        # Draw initial objects
        self.player = self.canvas.create_rectangle(
            self.player_x, self.player_y,
            self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE,
            fill=WHITE, outline=BLACK, width=2
        )
        self.score_text = self.canvas.create_text(10, 10, text="Score: 0", anchor="nw", fill=WHITE, font=("Arial", 25))
        
        # Start game loop
        self.update()

    def create_spike(self, x):
        """Create a spike with shadow effect."""
        y = SCREEN_HEIGHT - GROUND_HEIGHT
        # Shadow (slightly offset)
        shadow = self.canvas.create_polygon(
            x + 2, y + 2,
            x + 22, y + 2,
            x + 12, y - 28,
            fill=DARK_RED, outline=BLACK
        )
        # Main spike
        spike = self.canvas.create_polygon(
            x, y,
            x + 15, y,
            x + 10, y - 30,
            fill=RED, outline=BLACK
        )
        return {"x": x, "shadow": shadow, "spike": spike}

    def jump(self, event):
        """Handle jump input."""
        if self.on_ground and not self.game_over:
            self.player_velocity_y = JUMP_VELOCITY
            self.on_ground = False

    def update(self):
        """Game loop."""
        try:
            if not self.game_over:
                # Update player
                self.player_velocity_y += GRAVITY
                self.player_y += self.player_velocity_y
                if self.player_y >= SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE:
                    self.player_y = SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE
                    self.player_velocity_y = 0
                    self.on_ground = True
                
                # Move player on canvas
                self.canvas.coords(
                    self.player,
                    self.player_x, self.player_y,
                    self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE
                )
                
                # Update spikes
                for spike in self.spikes[:]:
                    spike["x"] -= SCROLL_SPEED
                    self.canvas.move(spike["spike"], -SCROLL_SPEED, 0)
                    self.canvas.move(spike["shadow"], -SCROLL_SPEED, 0)
                    if spike["x"] < -20:
                        self.canvas.delete(spike["spike"])
                        self.canvas.delete(spike["shadow"])
                        self.spikes.remove(spike)
                        self.spikes.append(self.create_spike(SCREEN_WIDTH + random.randint(300, 550)))
                    
                    # Collision detection
                    if (abs(self.player_x - spike["x"]) < PLAYER_SIZE and
                        abs(self.player_y - (SCREEN_HEIGHT - GROUND_HEIGHT -5.8)) < PLAYER_SIZE):
                        self.game_over = True
                        self.canvas.create_text(
                            SCREEN_WIDTH // 2, SCREEN_HEIGHT // 2,
                            text="Game Over! Press R to Restart", fill=WHITE, font=("Minecraft", 30)
                        )
                
                self.score += 1
                self.canvas.itemconfig(self.score_text, text=f"Score: {self.score}")
            
            # Draw ground with checkered texture
            self.canvas.delete("ground")
            for x in range(0, SCREEN_WIDTH, 20):
                for y in range(SCREEN_HEIGHT - GROUND_HEIGHT, SCREEN_HEIGHT, 20):
                    color = GREEN if (x // 20 + y // 20) % 2 == 0 else DARK_GREEN
                    self.canvas.create_rectangle(
                        x, y, x + 20, y + 20, fill=color, outline="", tags="ground"
                    )
            
            # Restart on 'R' key
            if self.game_over:
                self.root.bind("<r>", self.restart)
            
            # Schedule next update
            self.root.after(FRAME_TIME, self.update)
        
        except Exception as e:
            print(f"Game loop error: {e}")
            self.root.quit()

    def restart(self, event):
        """Restart the game."""
        try:
            self.canvas.delete("all")
            self.player_x = 100
            self.player_y = SCREEN_HEIGHT - GROUND_HEIGHT - PLAYER_SIZE
            self.player_velocity_y = 0
            self.on_ground = True
            for spike in self.spikes:
                self.canvas.delete(spike["spike"])
                self.canvas.delete(spike["shadow"])
            self.spikes = [self.create_spike(SCREEN_WIDTH + i * 200) for i in range(4)]
            self.score = 0
            self.game_over = False
            self.player = self.canvas.create_rectangle(
                self.player_x, self.player_y,
                self.player_x + PLAYER_SIZE, self.player_y + PLAYER_SIZE,
                fill=WHITE, outline=BLACK, width=2
            )
            self.score_text = self.canvas.create_text(
                10, 10, text="Score: 0", anchor="nw", fill=WHITE, font=("Arial", 14)
            )
            self.root.bind("<r>", lambda e: None)  # Unbind restart
        except Exception as e:
            print(f"Restart error: {e}")

def main():
    try:
        root = tk.Tk()
        game = Game(root)
        root.mainloop()
    except Exception as e:
        print(f"Main error: {e}")
        sys.exit()

if __name__ == "__main__":
    main()